﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data;
using DataAccess;

/// <summary>
/// Patient treatment methods and proiperties
/// </summary>
public class CPatTreatment
{
    /// <summary>
    /// Default constructor
    /// </summary>
    public CPatTreatment()
    {
        //
        // TODO: Add constructor logic here
        //
    }


    /// <summary>
    /// update a treatment program record
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lTreatmentID"></param>
    /// <param name="lProgramID"></param>
    /// <param name="dtInitialDate"></param>
    /// <param name="dtEndDate"></param>
    /// <returns></returns>
    public bool UpdateTreatmentProgram(BaseMaster BaseMstr,
                                string strPatientID,
                                long lTreatmentID,
                                long lProgramID,
                                long lCPAID,
                                DateTime dtInitialDate,
                                DateTime dtEndDate)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nTreatmentID", lTreatmentID);
        plist.AddInputParameter("pi_nProgramID", lProgramID);
        plist.AddInputParameter("pi_nCPAID", lCPAID);

        plist.AddInputParameter("pi_dtInitialDate", dtInitialDate);
        plist.AddInputParameter("pi_dtEndDate", dtEndDate);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PAT_TREATMENT.UpdateTreatmentProgram",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {

            return true;
        }

        return false;
    }


    /// <summary>
    /// remove a treatment and its programs
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lTreatmentID"></param>
    /// <returns></returns>
    public bool DeleteTreatment(BaseMaster BaseMstr,
                                string strPatientID,
                                long lTreatmentID
                                )
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", BaseMstr.RegionID);
        plist.AddInputParameter("pi_nSiteID", BaseMstr.SiteID);
        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nTreatmentID", lTreatmentID);
        
        BaseMstr.DBConn.ExecuteOracleSP("PCK_PAT_TREATMENT.DeleteTreatment",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {

            return true;
        }

        return false;
    }

    /// <summary>
    /// delete a treatment program
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lProgramID"></param>
    /// <returns></returns>
    public bool DeleteTreatmentProgram(BaseMaster BaseMstr,
                               string strPatientID,
                               long lProgramID
                               )
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", BaseMstr.RegionID);
        plist.AddInputParameter("pi_nSiteID", BaseMstr.SiteID);
        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nProgramID", lProgramID);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PAT_TREATMENT.DeleteTreatmentProgram",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {

            return true;
        }

        return false;
    }

    /// <summary>
    /// update patient treatment 
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lTreatmentID"></param>
    /// <param name="dtInitialDate"></param>
    /// <param name="dtEndDate"></param>
    /// <returns></returns>
    public bool UpdateTreatment(BaseMaster BaseMstr,
                                string strPatientID,
                                long lTreatmentID,
                                long lTreatmentType,
                                string strTreatmentTitle,
                                DateTime dtInitialDate,
                                DateTime dtEndDate)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nTreatmentID", lTreatmentID);

        plist.AddInputParameter("pi_nTreatmentType", lTreatmentType);
        plist.AddInputParameter("pi_vTreatmentTitle", strTreatmentTitle);

        plist.AddInputParameter("pi_dtInitialDate", dtInitialDate);
        plist.AddInputParameter("pi_dtEndDate", dtEndDate);

    
        BaseMstr.DBConn.ExecuteOracleSP("PCK_PAT_TREATMENT.UpdateTreatment",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
         
            return true;
        }

        return false;
    }


    /// <summary>
    /// get patient treatment dataset
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <returns></returns>
    public DataSet GetTreatmentDS(BaseMaster BaseMstr,
                                  string strPatientID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PAT_TREATMENT.GetTreatmentRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// get treatment dataset by id
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lTreatmentID"></param>
    /// <returns></returns>
    public DataSet GetTreatmentDS(BaseMaster BaseMstr,
                                  string strPatientID,
                                  long lTreatmentID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nTreatmentID", lTreatmentID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PAT_TREATMENT.GetTreatmentRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// get dataset of treatment programs
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lTreatmentID"></param>
    /// <returns></returns>
    public DataSet GetTreatmentProgramDS(BaseMaster BaseMstr,
                                         string strPatientID,
                                         long lTreatmentID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nTreatmentID", lTreatmentID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PAT_TREATMENT.GetTreatmentProgramRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lTreatmentID"></param>
    /// <param name="lProgramID"></param>
    /// <returns></returns>
    public DataSet GetTreatmentProgramDS(BaseMaster BaseMstr,
                                         string strPatientID,
                                         long lTreatmentID,
                                         long lProgramID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_vPatientID", strPatientID);
        plist.AddInputParameter("pi_nTreatmentID", lTreatmentID);
        plist.AddInputParameter("pi_nProgramID", lProgramID);

        //call sp to get ds
        CDataSet cds = new CDataSet();
        DataSet ds = cds.GetOracleDataSet(BaseMstr.DBConn,
                                           "PCK_PAT_TREATMENT.GetTreatmentProgramRS",
                                            plist,
                                            out lStatusCode,
                                            out strStatusComment);


        if (lStatusCode == 0)
        {
            return ds;
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// insert a new treatment
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lTreatmentTypeID"></param>
    /// <param name="dtInitialDate"></param>
    /// <param name="dtEndDate"></param>
    /// <param name="strTitle"></param>
    /// <param name="lTreatmentID"></param>
    /// <returns></returns>
    public bool InsertTreatment(BaseMaster BaseMstr,
                                 string strPatientID,
                                 long lTreatmentTypeID,
                                 DateTime dtInitialDate,
                                 DateTime dtEndDate,
                                 string strTitle,
                                 out long lTreatmentID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //init treatment id
        lTreatmentID = -1;

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", BaseMstr.RegionID);
        plist.AddInputParameter("pi_nSiteID", BaseMstr.SiteID);
        plist.AddInputParameter("pi_vPatientID", strPatientID);

        plist.AddInputParameter("pi_nTreatmentTypeID", lTreatmentTypeID);
        plist.AddInputParameter("pi_dtInitialDate", dtInitialDate);
        plist.AddInputParameter("pi_dtEndDate", dtEndDate);

        plist.AddInputParameter("pi_vTreatmentTitle", strTitle);

        plist.AddOutputParameter("po_nPatTreatmentID", lTreatmentID);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PAT_TREATMENT.InsertTreatment",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            //get out params
            CDataParameter paramValue1 = plist.GetItemByName("po_nPatTreatmentID");
            lTreatmentID = paramValue1.LongParameterValue;

            return true;
        }

        return false;
    }

    /// <summary>
    /// insert a new treatment program
    /// </summary>
    /// <param name="BaseMstr"></param>
    /// <param name="strPatientID"></param>
    /// <param name="lTreatmentTypeID"></param>
    /// <param name="dtInitialDate"></param>
    /// <param name="dtEndDate"></param>
    /// <param name="strTitle"></param>
    /// <param name="lTreatmentID"></param>
    /// <returns></returns>
    public bool InsertTreatmentProgram(BaseMaster BaseMstr,
                                       string strPatientID,
                                       long lTreatmentID,
                                       long lCPAID,
                                       DateTime dtInitialDate,
                                       DateTime dtEndDate,
                                       out long lProgramID)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //init treatment id
        lProgramID = -1;

        //create a new parameter list with standard params from basemstr
        CDataParameterList plist = new CDataParameterList(BaseMstr.ASPSessionID,
                                                          BaseMstr.ClientIP,
                                                          BaseMstr.FXUserID);

        plist.AddInputParameter("pi_nRegionID", BaseMstr.RegionID);
        plist.AddInputParameter("pi_nSiteID", BaseMstr.SiteID);
        plist.AddInputParameter("pi_vPatientID", strPatientID);

        plist.AddInputParameter("pi_nPatTreatmentID", lTreatmentID);
        plist.AddInputParameter("pi_nCPAID", lCPAID);

        plist.AddInputParameter("pi_dtInitialDate", dtInitialDate);
        plist.AddInputParameter("pi_dtEndDate", dtEndDate);

        plist.AddOutputParameter("po_nPatTreatProgID", lProgramID);

        BaseMstr.DBConn.ExecuteOracleSP("PCK_PAT_TREATMENT.InsertTreatmentProgram",
                                        plist,
                                        out lStatusCode,
                                        out strStatusComment);

        if (lStatusCode == 0)
        {
            //get out params
            CDataParameter paramValue1 = plist.GetItemByName("po_nPatTreatProgID");
            lProgramID = paramValue1.LongParameterValue;

            return true;
        }

        return false;
    }

}